/* $Id: portal.c,v 1.5 2005/06/25 02:18:10 jwp Exp $
 *
 * Copyright 2005, PostgresPy Project.
 * http://python.projects.postgresql.org
 *
 *//*
 * imp/src/portal.c,v 1.5 2004/11/21 19:17:17 flaw
 * if/src/portal.c,v 1.10 2004/09/28 15:36:40 flaw
 *//*
 * Postgres Portal interface
 */
#include <postgres.h>
#include <access/heapam.h>
#include <catalog/pg_type.h>
#include <executor/executor.h>
#include <executor/execdesc.h>
#include <executor/tstoreReceiver.h>
#include <nodes/pg_list.h>
#include <nodes/params.h>
#include <tcop/dest.h>
#include <tcop/tcopprot.h>
#include <tcop/pquery.h>
#include <tcop/utility.h>
#include <utils/array.h>
#include <utils/palloc.h>
#include <utils/portal.h>
#include <utils/relcache.h>
#include <utils/tuplestore.h>
#include <pypg/postgres.h>
#include <pypg/environment.h>

#include <Python.h>
#include <structmember.h>
#include <pypg/python.h>

#include <pypg/externs.h>

#include <pypg/tupledesc.h>
#include <pypg/heaptuple.h>
#include <pypg/object.h>
#include <pypg/error.h>
#include <pypg/utils.h>

#include <pypg/query.h>
#include <pypg/call.h>
#include <pypg/call/portal.h>

static Portal
Portal_Initialize(PyObj self)
{
	volatile Portal port = NULL;
	volatile ParamListInfo pli = NULL;

	PG_TRY();
	{
		port = CreatePortal(PyPgPortal_FetchNameSTRING(self), true, true);
		port->cursorOptions = CURSOR_OPT_SCROLL;
		PortalDefineQuery(port,
			PyPgPortal_FetchQuerySTRING(self),
			"SELECT",
			list_make1(PyPgPortal_FetchQuery(self)),
			list_make1(PyPgPortal_FetchPlan(self)),
			CurrentMemoryContext
		);

		if (PyTuple_GET_SIZE(PyPgCall_FetchArguments(self)) > 0)
			pli = ParamListInfo_FromPyTuple(PyPgCall_FetchArguments(self));
		PortalStart(port, pli, GetLatestSnapshot());
	}
	PG_CATCH();
	{
		if (port)
		{
			PortalDrop(port, false);
			port = NULL;
		}
		PyErr_SetPgError();
	}
	PG_END_TRY();

	return(port);
}

static void
_R_Receive
(
#if PGV_MM == 80
 	HeapTuple ht, TupleDesc td,
#else
	TupleTableSlot *slot,
#endif
	DestReceiver *r
)
{
#if PGV_MM != 80
	HeapTuple ht = slot->tts_tuple;
#endif
	MemoryContext old;
	PyObj port, tup, tupd, tupl;

	port = (PyObj) ((void *) r - (void *) offsetof(struct PyPgPortal, p_rec));
	tupd = PyPgCall_FetchExpected(port);

	old = MemoryContextSwitchTo(PythonMemoryContext);

	ht = heap_copytuple(ht);
	tup = PyPgHeapTuple_New(tupd, ht);
	tupl = PyPgCall_FetchReturned(port);
	if (tupl == Py_None)
	{
		Py_DECREF(Py_None);
		tupl = NULL;
	}

	if (tupl)
		PyList_Append(tupl, tup);
	else
		PyPgCall_FixReturned(port, tup);

	MemoryContextSwitchTo(old);
}
static void _R_Startup(DestReceiver *r, int op, TupleDesc td) {}
static void _R_Shutdown(DestReceiver *r) {}
static void _R_Destroy(DestReceiver *r) {}

static void
_R_InitReceiver(DestReceiver *r)
{
#if PGV_MM == 80
	r->receiveTuple
#else
	r->receiveSlot
#endif
							= _R_Receive;
	r->rStartup			= _R_Startup;
	r->rShutdown		= _R_Shutdown;
	r->rDestroy			= _R_Destroy;
	r->mydest			= 0xFFA0;
}

static PyMemberDef PyPgPortal_Members[] = {
	{"query", T_OBJECT, offsetof(struct PyPgPortal, call_func), RO,
	"the query from which the Postgres.Portal was created"},
	{"name", T_OBJECT, offsetof(struct PyPgPortal, p_name), RO,
	"the name of the Portal that this Postgres.Portal created"},
	{NULL}
};

static void
port_closedError(PyObj self)
{
	/* Portal p = PyPgPortal_FetchPortal(self); */
	PyErr_Format(PyExc_ValueError, "operation on closed Postgres.Portal");
}
#define RaiseClosedPortalError(SELF, RET) do { \
	if (PyPgPortal_IsClosed(SELF)) { \
		port_closedError(SELF); \
		return(RET); \
	} \
} while(0)

static PyObj
_port_read(PyObj self, long q)
{
	volatile PyObj rob = NULL;
	Portal p;
	DestReceiver *r;
	p = PyPgPortal_FetchPortal(self);
	r = PyPgPortal_FetchReceiver(self);
	if (q != 0)
		PgError_TRAP(PortalRunFetch(p, FETCH_FORWARD, q, r));

	if (!PyErr_Occurred())
	{
		rob = PyPgCall_FetchReturned(self);
		PyPgCall_FixReturned(self, Py_None);
		Py_INCREF(Py_None);
	}
	return(rob);
}

static PyObj
port_read(PyObj self, PyObj args, PyObj kw)
{
	PyObj quantity = NULL, rob = NULL;
	long q;

	RaiseClosedPortalError(self, NULL);
	if (!PyArg_ParseTuple(args, "|O", &quantity))
		return(NULL);
	
	if (quantity != Py_None && quantity != NULL)
		q = PyInt_AsLong(quantity);
	else
		q = FETCH_ALL;

	if (PyPgPortal_AtEnd(self) && q > 0)
		return(PyList_New(0));
	else if (PyPgPortal_AtGenesis(self) && q < 0)
		return(PyList_New(0));

	PyPgCall_FixReturned(self, PyList_New(0));
	rob = _port_read(self, q);
	return(rob);
}

static long
_port_seek(PyObj self, long offset)
{
	long seeked;
	Portal p = PyPgPortal_FetchPortal(self);
	seeked = PortalRunFetch(p, FETCH_ABSOLUTE, offset, None_Receiver);
	return(seeked);
}

static PyObj
port_seek(PyObj self, PyObj args, PyObj kw)
{
	long offset, pos;

	RaiseClosedPortalError(self, NULL);
	if (!PyArg_ParseTuple(args, "i", &offset))
		return(NULL);

	pos = _port_seek(self, offset);
	return(PyInt_FromLong(pos));
}

static PyObj
port_tell(PyObj self)
{
	PyObj rob;
	RaiseClosedPortalError(self, NULL);
	rob = PyInt_FromLong(PyPgPortal_FetchPosition(self));
	return(rob);
}

static PyObj
port_close(PyObj self)
{
	Portal p = PyPgPortal_FetchPortal(self);
	RaiseClosedPortalError(self, NULL);

	PortalDrop(p, false);
	PyPgPortal_FixPortal(self, NULL);

	RETURN_NONE;
}

static PyMethodDef PyPgPortal_Methods[] = {
	{"read", (PyCFunction) port_read, METH_VARARGS,
		"read the given number of tuples from the portal"},
	{"seek", (PyCFunction) port_seek, METH_VARARGS,
		"seek to the given tuple offset of the portal"},
	{"tell", (PyCFunction) port_tell, METH_NOARGS,
		"return the current tuple offset"},
	{"close", (PyCFunction) port_close, METH_NOARGS,
		"close the portal to prevent further use"},
	{NULL}
};

static int
port_length(PyObj self)
{
	int seeked, state;
	state = PyPgPortal_FetchPosition(self);
	seeked = _port_seek(self, FETCH_ALL);
	_port_seek(self, state);
	return(seeked);
}

static PyObj
port_item(PyObj self, int index)
{
	long curpos = PyPgPortal_FetchPosition(self), seeked;
	PyObj rob = NULL;

	seeked = _port_seek(self, index);
	if (seeked != index)
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.Portal index, \"%d\", out of range", index);
	}
	else
		rob = _port_read(self, 1);

	_port_seek(self, curpos);
	return(rob);
}

static PyObj
port_slice(PyObj self, int from, int to)
{
	long curpos = PyPgPortal_FetchPosition(self), seeked;
	PyObj rob = NULL;

	seeked = _port_seek(self, from);
	if (seeked != from)
	{
		PyErr_Format(PyExc_IndexError,
			"Postgres.Portal index, \"%d\", out of range", from);
	}
	else
	{
		PyPgCall_FixReturned(self, PyList_New(0));
		rob = _port_read(self, to - from);
	}

	_port_seek(self, curpos);
	return(rob);

	RETURN_NONE;
}

static PySequenceMethods PyPgPortalAsSequence = {
	port_length,	/* sq_length */
	NULL,				/* sq_concat */
	NULL,				/* sq_repeat */
	port_item,		/* sq_item */
	port_slice,		/* sq_slice */
	NULL,				/* sq_ass_item */
};

static PyObj
port_iternext(PyObj self)
{
	PyObj rob = NULL;

	RaiseClosedPortalError(self, NULL);

	if (!PyPgPortal_AtEnd(self))
		rob = _port_read(self, 1);
	return(rob);
}

static void
port_dealloc(PyObj self)
{
	PyObj ob;

	if (!PyPgPortal_IsClosed(self))
		port_close(self);

	ob = PyPgPortal_FetchName(self);
	DECREF(ob);
	PyPgPortal_FixName(self, NULL);

	self->ob_type->tp_base->tp_dealloc(self);
}

static PyObj
port_call(PyObj self, PyObj args, PyObj kw)
{
	PyObj rob;
	rob = self->ob_type->tp_iternext(self);
	return(rob);
}

static char PyPgPortal_Doc[] = "Postgres Portal";
PyTypeObject PyPgPortal_Type = {
	PyObject_HEAD_INIT(NULL)
	0,										/* ob_size */
	"Postgres.Portal",				/* tp_name */
	sizeof(struct PyPgPortal),		/* tp_basicsize */
	0,										/* tp_itemsize */
	port_dealloc,						/* tp_dealloc */
	NULL,									/* tp_print */
	NULL,									/* tp_getattr */
	NULL,									/* tp_setattr */
	NULL,									/* tp_compare */
	NULL,									/* tp_repr */
	NULL,									/* tp_as_number */
	&PyPgPortalAsSequence,			/* tp_as_sequence */
	NULL,									/* tp_as_mapping */
	NULL,									/* tp_hash */
	port_call,							/* tp_call */
	NULL,									/* tp_str */
	NULL,									/* tp_getattro */
	NULL,									/* tp_setattro */
	NULL,									/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT,			   /* tp_flags */
	(char *) PyPgPortal_Doc,		/* tp_doc */
	NULL,									/* tp_traverse */
	NULL,									/* tp_clear */
	NULL,									/* tp_richcompare */
	0,										/* tp_weaklistoffset */
	(getiterfunc) Py_RETURN_SELF,	/* tp_iter */
	port_iternext,						/* tp_iternext */
	PyPgPortal_Methods,				/* tp_methods */
	PyPgPortal_Members,				/* tp_members */
	NULL,									/* tp_getset */
	&PyPgCall_Type,					/* tp_base */
	NULL,									/* tp_dict */
	NULL,									/* tp_descr_get */
	NULL,									/* tp_descr_set */
	0,										/* tp_dictoffset */
	NULL,									/* tp_init */
	NULL,									/* tp_alloc */
	NULL,									/* tp_new */
};

/*
 * PyPgPortal_New
 * 	Create a portal object based on a query, plan, and parameters.
 */
PyObj
PyPgPortal_Initialize(PyObj self, PyObj query, PyObj args, Query *q, Plan *p)
{
	PyObj tdo, pname;
	TupleDesc td;
	Portal port;

	if (self == NULL) return(NULL);

	PyPgCall_FixReturned(self, Py_None);
	Py_INCREF(Py_None);
	PyPgCall_FixKeywords(self, Py_None);
	Py_INCREF(Py_None);

	PyPgCall_FixFunction(self, query);
	INCREF(query);
	PyPgCall_FixArguments(self, args);
	INCREF(args);
	_R_InitReceiver(PyPgPortal_FetchReceiver(self));

	pname = PyString_FromFormat("<Python Portal %p>", self);
	if (pname == NULL)
	{
		PyErr_SetString(PyExc_MemoryError,
			"failed to allocate memory for Portal name");
		goto DECREF_args;
	}
	PyPgPortal_FixName(self, pname);

	PyPgPortal_FixQuery(self, q);
	PyPgPortal_FixPlan(self, p);
	port = Portal_Initialize(self);
	if (port == NULL) goto DECREF_pname;
	PyPgPortal_FixPortal(self, port);

	td = CreateTupleDescCopy(port->tupDesc);
	tdo = PyPgTupleDesc_New(td);
	if (tdo == NULL)
	{
		FreeTupleDesc(td);
		goto DECREF_pname;
	}
	PyPgCall_FixExpected(self, tdo);

	return(self);
DECREF_pname:
	DECREF(pname);
DECREF_args:
	DECREF(args);
	DECREF(query);
	/*
	 * Returned, and keywords
	 */
	Py_DECREF(Py_None);
	Py_DECREF(Py_None);
	self->ob_type->tp_free(self);
	return(NULL);
}
/*
 * vim: ts=3:sw=3:noet:
 */
